using System;
using System.Xml;

using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary.Messages
{
	/// <summary>
	/// This class represents VistALink protocol simple response 
	/// messages as defined in vlSimpleResponse.xsd
	/// </summary>
	public abstract class VistALinkSimpleResponseMessage : BaseResponseMessage
	{	
		/// <summary>
		/// VistALink message type string. 
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string MessageType = "gov.va.med.foundations.vistalink.system.response";

		/// <summary>
		/// VistALink message XSD schema file name string. 		
		/// This constant is made public to recognize message XML.
		/// </summary>
		public const string XsdName = "vlSimpleResponse.xsd";

		/// <summary>
		/// Constructor specifying minimal required VistALink simple response message parameters.
		/// </summary>
		/// <param name="responseType">VistALink simple response message type</param>
		/// <param name="responseStatus">VistALink response status</param>
		protected VistALinkSimpleResponseMessage( string responseType, ResponseStatus responseStatus ) : 
			base( MessageType, XsdName, responseType, responseStatus ) 
		{
			if( !IsValidResponseStatus( _responseStatus ) )
				throw( new ArgumentOutOfRangeException( SR.Exceptions.VlSimpleResponseMsgWrongResponseStatus( responseStatus.ToString() ) ) );
		}

		/// <summary>
		/// Standard VistALink message deserialization constructor. 
		/// Allows recreating message object from XML document.
		/// </summary>
		/// <param name="responseType">VistALink simple response message type</param>
		/// <param name="sourceDoc">Source XML document</param>
		protected VistALinkSimpleResponseMessage( string responseType, XmlDocument sourceDoc ) : 
			base( MessageType, XsdName, responseType, sourceDoc )
		{
			if( !IsValidResponseStatus( _responseStatus ) )
				throw( new XmlParseException( SR.Exceptions.VlSimpleResponseMsgWrongResponseStatus( _responseStatus.ToString() ) ) );
		}

		/// <summary>
		/// This method checks if supplied response status falls into acceptable range for
		/// simple VistALink response message. 
		/// </summary>
		/// <param name="status">Response status to check.</param>
		/// <returns>True if response status is within acceptable range, false otherwise.</returns>
		private bool IsValidResponseStatus( ResponseStatus status )
		{
			return status == ResponseStatus.Success || status == ResponseStatus.Failure;
		}
	}
}
